import { useRef, useState } from "react";
import { toBlob } from "html-to-image";
import JSZip from "jszip";
import {
  Bar,
  BarChart,
  CartesianGrid,
  Cell,
  Pie,
  PieChart,
  XAxis,
  YAxis,
} from "recharts";
import {
  type ChartConfig,
  ChartContainer,
  ChartLegend,
  ChartLegendContent,
  ChartTooltip,
  ChartTooltipContent,
} from "@/components/ui/chart";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Download, FileArchive, Loader2 } from "lucide-react";

import {
  colors,
  fundingRegionalData,
  marketProjectionData,
  perceptionConcernsData,
  consumerSentimentData,
  regulationData,
  employmentData,
  europeAdoptionData,
  chatgptUsesData,
  voiceAssistantsData,
  securityData,
  rawCsvFiles,
  chartMetadata,
} from "./data";

// Chart configs
const fundingRegionalConfig = {
  companies: { label: "Companies" },
  "United States": { label: "United States", color: colors.blue },
  "European Union": { label: "European Union", color: colors.red },
  "United Kingdom": { label: "United Kingdom", color: colors.green },
} satisfies ChartConfig;

const marketProjectionConfig = {
  value: { label: "Market Value (Billion USD)", color: colors.indigo },
} satisfies ChartConfig;

const perceptionConcernsConfig = {
  value: { label: "Percentage" },
  "Inaccuracy Concerns": { label: "Inaccuracy", color: colors.red },
  "IP Violation Worries": { label: "IP Violations", color: colors.yellow },
  "Security Concerns": { label: "Security", color: colors.blue },
} satisfies ChartConfig;

const consumerSentimentConfig = {
  percentage: { label: "Percentage (%)" },
  "Feel AI is Overhyped": { label: "Overhyped", color: colors.orange },
  "Trust Has Grown Since GenAI": { label: "Trust Grown", color: colors.teal },
} satisfies ChartConfig;

const regulationConfig = {
  percentage: { label: "Percentage (%)", color: colors.purple },
} satisfies ChartConfig;

const employmentConfig = {
  created: { label: "Jobs Created (M)", color: colors.green },
  eliminated: { label: "Jobs Eliminated (M)", color: colors.red },
} satisfies ChartConfig;

const europeAdoptionConfig = {
  adoption: { label: "Adoption Rate (%)", color: colors.cyan },
} satisfies ChartConfig;

const chatgptUsesConfig = {
  percentage: { label: "Percentage (%)", color: colors.indigo },
} satisfies ChartConfig;

const voiceAssistantsConfig = {
  value: { label: "Users (Millions)" },
  "Amazon Alexa": { label: "Amazon Alexa", color: "#FF9900" },
  "Apple Siri": { label: "Apple Siri", color: "#A2AAAD" },
  "Google Assistant": { label: "Google Assistant", color: "#4285F4" },
} satisfies ChartConfig;

const securityConfig = {
  percentage: { label: "Percentage (%)" },
  "Using AI for Fraud Detection": { label: "Fraud Detection", color: colors.green },
  "Threat Detection Improvement": { label: "Threat Detection", color: colors.blue },
} satisfies ChartConfig;

export default function AIDataDashboard() {
  const [isDownloading, setIsDownloading] = useState(false);
  const [isDownloadingSource, setIsDownloadingSource] = useState(false);

  const chartRefs = useRef<(HTMLDivElement | null)[]>([]);

  const generateCombinedCSV = (): string => {
    const rows: string[] = ["Chart,Category,Metric,Value,Unit"];

    fundingRegionalData.forEach((d) => {
      rows.push(`"Funding by Region","${d.region}","Companies","${d.companies}","count"`);
    });

    marketProjectionData.forEach((d) => {
      rows.push(`"Market Projection","${d.year}","Market Value","${d.value}","billion USD"`);
    });

    perceptionConcernsData.forEach((d) => {
      rows.push(`"Perception Concerns","Concerns","${d.name}","${d.value}","%"`);
    });

    consumerSentimentData.forEach((d) => {
      rows.push(`"Consumer Sentiment","Sentiment","${d.sentiment}","${d.percentage}","%"`);
    });

    regulationData.forEach((d) => {
      rows.push(`"AI Regulation","Demands","${d.demand}","${d.percentage}","%"`);
    });

    employmentData.forEach((d) => {
      rows.push(`"Employment Impact","${d.category}","Jobs Created","${d.created}","millions"`);
      rows.push(`"Employment Impact","${d.category}","Jobs Eliminated","${d.eliminated}","millions"`);
    });

    europeAdoptionData.forEach((d) => {
      rows.push(`"EU Adoption","${d.country}","Adoption Rate","${d.adoption}","%"`);
    });

    chatgptUsesData.forEach((d) => {
      rows.push(`"ChatGPT Uses","Business","${d.useCase}","${d.percentage}","%"`);
    });

    voiceAssistantsData.forEach((d) => {
      rows.push(`"Voice Assistants","Users","${d.name}","${d.value}","millions"`);
    });

    securityData.forEach((d) => {
      rows.push(`"AI Security","Cybersecurity","${d.metric}","${d.percentage}","%"`);
    });

    return rows.join("\n");
  };

  const downloadAll = async () => {
    setIsDownloading(true);
    try {
      const zip = new JSZip();
      const chartsFolder = zip.folder("charts");

      for (let i = 0; i < chartRefs.current.length; i++) {
        const ref = chartRefs.current[i];
        if (ref && chartsFolder) {
          const blob = await toBlob(ref, { pixelRatio: 2 });
          if (blob) {
            chartsFolder.file(`${chartMetadata[i].id}.png`, blob);
          }
        }
      }

      const csvContent = generateCombinedCSV();
      zip.file("ai-dashboard-data.csv", csvContent);

      const content = await zip.generateAsync({ type: "blob" });
      const url = URL.createObjectURL(content);
      const a = document.createElement("a");
      a.href = url;
      a.download = "ai-dashboard-export.zip";
      a.click();
      URL.revokeObjectURL(url);
    } finally {
      setIsDownloading(false);
    }
  };

  const downloadSource = async () => {
    setIsDownloadingSource(true);
    try {
      const zip = new JSZip();

      Object.entries(rawCsvFiles).forEach(([filename, content]) => {
        zip.file(filename, content);
      });

      const content = await zip.generateAsync({ type: "blob" });
      const url = URL.createObjectURL(content);
      const a = document.createElement("a");
      a.href = url;
      a.download = "ai-dashboard-source.zip";
      a.click();
      URL.revokeObjectURL(url);
    } finally {
      setIsDownloadingSource(false);
    }
  };

  return (
    <div className="grid gap-6">
      <div className="flex gap-3">
        <Button onClick={downloadAll} disabled={isDownloading || isDownloadingSource}>
          {isDownloading ? (
            <Loader2 className="h-4 w-4 animate-spin" />
          ) : (
            <Download className="h-4 w-4" />
          )}
          Download All
        </Button>
        <Button
          variant="outline"
          onClick={downloadSource}
          disabled={isDownloading || isDownloadingSource}
        >
          {isDownloadingSource ? (
            <Loader2 className="h-4 w-4 animate-spin" />
          ) : (
            <FileArchive className="h-4 w-4" />
          )}
          Download Source
        </Button>
      </div>

      {/* Chart 1: Funding Regional */}
      <Card ref={(el) => { chartRefs.current[0] = el; }}>
        <CardHeader>
          <CardTitle>Newly Funded AI Companies by Region (2022)</CardTitle>
          <CardDescription>AI companies receiving funding over $1.5M</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={fundingRegionalConfig}>
            <BarChart data={fundingRegionalData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="region" tickLine={false} tickMargin={10} axisLine={false} />
              <YAxis />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="companies" radius={4}>
                {fundingRegionalData.map((entry) => (
                  <Cell key={entry.region} fill={entry.fill} />
                ))}
              </Bar>
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 2: AI Market Projection */}
      <Card ref={(el) => { chartRefs.current[1] = el; }}>
        <CardHeader>
          <CardTitle>Global AI Market Value: 2023 vs 2030</CardTitle>
          <CardDescription>Expected to grow to $1.8 trillion by 2030</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={marketProjectionConfig}>
            <BarChart data={marketProjectionData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="year" tickLine={false} tickMargin={10} axisLine={false} />
              <YAxis tickFormatter={(value) => `$${value}B`} />
              <ChartTooltip
                content={<ChartTooltipContent formatter={(value) => `$${value}B`} />}
              />
              <Bar dataKey="value" radius={4}>
                {marketProjectionData.map((entry) => (
                  <Cell key={entry.year} fill={entry.fill} />
                ))}
              </Bar>
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 3: Perception Concerns (Pie) */}
      <Card ref={(el) => { chartRefs.current[2] = el; }}>
        <CardHeader>
          <CardTitle>Primary Concerns Preventing AI Adoption</CardTitle>
          <CardDescription>Top concerns cited by organizations</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={perceptionConcernsConfig} className="mx-auto aspect-square max-h-[350px]">
            <PieChart>
              <ChartTooltip content={<ChartTooltipContent nameKey="name" />} />
              <Pie
                data={perceptionConcernsData}
                dataKey="value"
                nameKey="name"
                cx="50%"
                cy="50%"
                outerRadius={120}
                label={({ value }) => `${value}%`}
              />
              <ChartLegend content={<ChartLegendContent nameKey="name" />} />
            </PieChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 4: Consumer Sentiment */}
      <Card ref={(el) => { chartRefs.current[3] = el; }}>
        <CardHeader>
          <CardTitle>Consumer Sentiment on AI</CardTitle>
          <CardDescription>Public perception of AI technology</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={consumerSentimentConfig}>
            <BarChart data={consumerSentimentData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="sentiment" tickLine={false} tickMargin={10} axisLine={false} hide />
              <YAxis tickFormatter={(value) => `${value}%`} />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="percentage" radius={4}>
                {consumerSentimentData.map((entry) => (
                  <Cell key={entry.sentiment} fill={entry.fill} />
                ))}
              </Bar>
              <ChartLegend content={<ChartLegendContent nameKey="sentiment" />} />
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 5: AI Regulation */}
      <Card ref={(el) => { chartRefs.current[4] = el; }}>
        <CardHeader>
          <CardTitle>Public Demands for AI Regulation</CardTitle>
          <CardDescription>Support for AI oversight and transparency</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={regulationConfig}>
            <BarChart data={regulationData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="demand" tickLine={false} tickMargin={10} axisLine={false} hide />
              <YAxis tickFormatter={(value) => `${value}%`} />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="percentage" radius={4}>
                {regulationData.map((entry) => (
                  <Cell key={entry.demand} fill={entry.fill} />
                ))}
              </Bar>
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 6: Employment Impact */}
      <Card ref={(el) => { chartRefs.current[5] = el; }}>
        <CardHeader>
          <CardTitle>AI Impact on Global Employment by 2025</CardTitle>
          <CardDescription>Jobs created vs eliminated (in millions)</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={employmentConfig}>
            <BarChart data={employmentData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="category" tickLine={false} tickMargin={10} axisLine={false} />
              <YAxis />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="created" fill={colors.green} radius={4} />
              <Bar dataKey="eliminated" fill={colors.red} radius={4} />
              <ChartLegend content={<ChartLegendContent />} />
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 7: Europe AI Adoption */}
      <Card ref={(el) => { chartRefs.current[6] = el; }}>
        <CardHeader>
          <CardTitle>AI Adoption Rates Across EU Countries (2024)</CardTitle>
          <CardDescription>Percentage of enterprises using AI technologies</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={europeAdoptionConfig} className="h-[300px]">
            <BarChart data={europeAdoptionData} layout="vertical">
              <CartesianGrid horizontal={false} />
              <XAxis type="number" tickFormatter={(value) => `${value}%`} />
              <YAxis dataKey="country" type="category" tickLine={false} axisLine={false} width={80} />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="adoption" fill={colors.cyan} radius={4} />
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 8: ChatGPT Business Uses */}
      <Card ref={(el) => { chartRefs.current[7] = el; }}>
        <CardHeader>
          <CardTitle>Anticipated ChatGPT Business Use Cases</CardTitle>
          <CardDescription>Expected applications in business operations</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={chatgptUsesConfig} className="h-[350px]">
            <BarChart data={chatgptUsesData} layout="vertical">
              <CartesianGrid horizontal={false} />
              <XAxis type="number" tickFormatter={(value) => `${value}%`} />
              <YAxis dataKey="useCase" type="category" tickLine={false} axisLine={false} width={130} />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="percentage" fill={colors.indigo} radius={4} />
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 9: Voice Assistants (Pie) */}
      <Card ref={(el) => { chartRefs.current[8] = el; }}>
        <CardHeader>
          <CardTitle>US Voice Assistant Users (2022)</CardTitle>
          <CardDescription>Active users in millions</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={voiceAssistantsConfig} className="mx-auto aspect-square max-h-[350px]">
            <PieChart>
              <ChartTooltip content={<ChartTooltipContent nameKey="name" />} />
              <Pie
                data={voiceAssistantsData}
                dataKey="value"
                nameKey="name"
                cx="50%"
                cy="50%"
                outerRadius={120}
                label={({ value }) => `${value}M`}
              />
              <ChartLegend content={<ChartLegendContent nameKey="name" />} />
            </PieChart>
          </ChartContainer>
        </CardContent>
      </Card>

      {/* Chart 10: AI Security */}
      <Card ref={(el) => { chartRefs.current[9] = el; }}>
        <CardHeader>
          <CardTitle>AI in Cybersecurity</CardTitle>
          <CardDescription>AI adoption in security operations</CardDescription>
        </CardHeader>
        <CardContent>
          <ChartContainer config={securityConfig}>
            <BarChart data={securityData}>
              <CartesianGrid vertical={false} />
              <XAxis dataKey="metric" tickLine={false} tickMargin={10} axisLine={false} hide />
              <YAxis tickFormatter={(value) => `${value}%`} />
              <ChartTooltip content={<ChartTooltipContent />} />
              <Bar dataKey="percentage" radius={4}>
                {securityData.map((entry) => (
                  <Cell key={entry.metric} fill={entry.fill} />
                ))}
              </Bar>
              <ChartLegend content={<ChartLegendContent nameKey="metric" />} />
            </BarChart>
          </ChartContainer>
        </CardContent>
      </Card>
    </div>
  );
}
